const { default: slugify } = require('slugify');
const { Product, productValidate } = require('../../models/product/product');
const { deleteManyOldImage } = require('../../utils/deleteOldImage');
const {
  idForSlug,
  generateSecureUniqueId,
} = require('../../utils/idGenerator');


exports.productAllList = async (req, res) => {
  try {
    const {
      sortBy,
      order,
      category,
      color,
      brand,
      gender,
      recommended,
      bestSeller,
      stock,
      campaign,
    } = req.query;

    let filter = {};

    if (category) filter.category = category;
    if (color) filter.color = color;
    if (brand) filter.brand = brand;
    if (gender) filter.gender = gender;
    if (campaign) filter.campaign = campaign;

    if (recommended !== undefined) {
      filter.recommended = recommended === "true";
    }
    if (bestSeller !== undefined) {
      filter.bestSeller = bestSeller === "true";
    }

    // Stock filter
    if (stock !== undefined) {
      const stockNum = Number(stock);
      if (!isNaN(stockNum)) {
        if (stockNum === 0) {
          filter.stock = 0; // yalnız stok 0 olanlar
        } else {
          filter.stock = { $gt: stockNum }; // verilən rəqəmdən böyük stok
        }
      }
    }

    // Sort
    let sort = {};
    if (sortBy) {
      const sortField =
        sortBy === "titleAz"
          ? "titleAz"
          : sortBy === "price"
          ? "price"
          : null;
      if (sortField) {
        sort[sortField] = order === "desc" ? -1 : 1;
      }
    }

    const products = await Product.find(filter).sort(sort);
    res.json(products);

  } catch (err) {
    console.error(err);
    res.status(500).json({ message: "Server xətası" });
  }
};

exports.productFindBySlug = async (req, res) => {
  try {
    

    const product = await Product.findOne({ slug:req.params.slug });

    if (!product) {
      return res.status(404).json({
        az: 'Məhsul tapılmadı.',
        en: 'Product not found.',
        ru: 'Товар не найден.',
      });
    }

    res.status(200).json(product);
  } catch (error) {
    console.error('Slug ilə məhsul axtarışında xəta:', error);
    res.status(500).json({
      az: 'Server xətası baş verdi.',
      en: 'Server error occurred.',
      ru: 'Произошла ошибка сервера.',
    });
  }
};

exports.productAdd = async (req, res) => {
  const { error } = productValidate(req.body);
  if (error) {
    return res.status(400).send(error.message);
  }

  let filesObj = req.files;
  let filesObjLength = Object.keys(filesObj).length;

  let product = new Product(req.body);

  const lastProduct = await Product.findOne().sort({ row: -1 });
  product.row = lastProduct ? lastProduct.row + 1 : 1;

  product.slug = slugify(req.body.titleEn, { lower: true }) + '-' + idForSlug();
  product.productID = generateSecureUniqueId();

  if (filesObjLength > 0) {
    product.images = req.files.images
      ? req.files.images.map((item) => item.path)
      : [];
    product.thumbnail = req.files.thumbnail
      ? req.files.thumbnail.map((item) => item.path)
      : [];
  }

  const result = await product.save();
  res.status(201).json({
    message: {
      az: 'Məlumat uğurla əlavə edildi.',
      en: 'Information added successfully.',
      ru: 'Информация успешно добавлена.',
    },
    data: result,
  });
};

exports.productEdit = async (req, res) => {
  const { error } = productValidate(req.body);
  if (error) {
    res.status(400).send(error.message);
  } else {
    let product;
    product = await Product.findById(req.params.id);

    if (!product) {
      return res.status(404).json({
        az: 'Belə bir məlumat yoxdur.',
        en: 'There is no such data.',
        ru: 'Таких данных нет.',
      });
    } else {
      let filesObj = req.files;
      filesObjLength = Object.keys(filesObj).length;
      if (filesObjLength === 0) {
        product = await Product.findByIdAndUpdate(req.params.id, {
          ...req.body,
        });

        await product.save();

        res.status(200).json({
          message: {
            az: 'Məlumat uğurla yeniləndi.',
            en: 'Information updated successfully.',
            ru: 'Информация успешно обновлена.',
          },
          data: product,
        });
      } else {
        product = await Product.findByIdAndUpdate(req.params.id, {
          ...req.body,
        });

        deleteManyOldImage(product.images);
        deleteManyOldImage(product.thumbnail);

        const uploadFile = [];
        req.files.images &&
          req.files.images.map(async (item) => {
            uploadFile.push(item.path);
          });
        product.images = uploadFile;

        const uploadCoverFile = [];
        req.files.thumbnail &&
          req.files.thumbnail.map(async (item) => {
            uploadCoverFile.push(item.path);
          });
        product.thumbnail = uploadCoverFile;

        await product.save();

        res.status(200).json({
          message: {
            az: 'Məlumat uğurla yeniləndi.',
            en: 'Information updated successfully.',
            ru: 'Информация успешно обновлена.',
          },
          data: product,
        });
      }
    }
  }
};

exports.productDelete = async (req, res) => {
  const product = await Product.findByIdAndDelete(req.params.id);
  if (!product) {
    return res.status(404).json({
      az: 'Belə bir məlumat yoxdur.',
      en: 'There is no such data.',
      ru: 'Таких данных нет.',
    });
  }
  deleteManyOldImage(product.thumbnail);
  deleteManyOldImage(product.images);
  res.status(200).json({
    message: {
      az: 'Məlumat uğurla silindi.',
      en: 'The information was deleted successfully.',
      ru: 'Информация успешно удалена.',
    },
    data: product,
  });
};
